<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
requireLogin();

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    // Get form data
    $quotation_id = intval($_POST['quotation_id'] ?? 0);
    $customer_id = !empty($_POST['customer_id']) ? intval($_POST['customer_id']) : null;
    $valid_until = $_POST['valid_until'] ?? '';
    $notes = $_POST['notes'] ?? '';
    $product_ids = $_POST['product_id'] ?? [];
    $quantities = $_POST['quantity'] ?? [];
    $unit_prices = $_POST['unit_price'] ?? [];
    
    // Validate required fields
    if ($quotation_id <= 0) {
        throw new Exception('Invalid quotation ID');
    }
    
    if (empty($valid_until)) {
        throw new Exception('Valid until date is required');
    }
    
    if (empty($product_ids) || empty($quantities) || empty($unit_prices)) {
        throw new Exception('At least one product item is required');
    }
    
    // Validate arrays have same length
    if (count($product_ids) !== count($quantities) || count($product_ids) !== count($unit_prices)) {
        throw new Exception('Invalid product data');
    }
    
    // Check if quotation exists and get current details
    $stmt = $pdo->prepare("
        SELECT quote_number, status, created_by 
        FROM quotations 
        WHERE id = ?
    ");
    $stmt->execute([$quotation_id]);
    $quotation = $stmt->fetch();
    
    if (!$quotation) {
        throw new Exception('Quotation not found');
    }
    
    // Check if user can edit this quotation
    $stmt = $pdo->prepare("SELECT is_super_admin FROM roles WHERE id = (SELECT role_id FROM users WHERE id = ?)");
    $stmt->execute([$_SESSION['user_id']]);
    $isSuperAdmin = $stmt->fetchColumn();
    
    if (!$isSuperAdmin && $quotation['created_by'] != $_SESSION['user_id']) {
        throw new Exception('You do not have permission to edit this quotation');
    }
    
    // Check if quotation can be edited
    if ($quotation['status'] === 'converted') {
        throw new Exception('Cannot edit a quotation that has been converted to a sale');
    }
    
    // Validate individual items
    $items = [];
    $total_amount = 0;
    
    for ($i = 0; $i < count($product_ids); $i++) {
        $product_id = intval($product_ids[$i]);
        $quantity = floatval($quantities[$i]);
        $unit_price = floatval($unit_prices[$i]);
        
        if ($product_id <= 0 || $quantity <= 0 || $unit_price <= 0) {
            throw new Exception('Invalid product, quantity, or price in item ' . ($i + 1));
        }
        
        $total_price = $quantity * $unit_price;
        $total_amount += $total_price;
        
        $items[] = [
            'product_id' => $product_id,
            'quantity' => $quantity,
            'unit_price' => $unit_price,
            'total_price' => $total_price
        ];
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Update quotation
        $stmt = $pdo->prepare("
            UPDATE quotations 
            SET customer_id = ?, total_amount = ?, notes = ?, valid_until = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $customer_id,
            $total_amount,
            $notes,
            $valid_until,
            $quotation_id
        ]);
        
        // Delete existing quotation items
        $stmt = $pdo->prepare("DELETE FROM quotation_items WHERE quotation_id = ?");
        $stmt->execute([$quotation_id]);
        
        // Insert new quotation items
        $stmt = $pdo->prepare("
            INSERT INTO quotation_items (quotation_id, product_id, quantity, unit_price, total_price)
            VALUES (?, ?, ?, ?, ?)
        ");
        
        foreach ($items as $item) {
            $stmt->execute([
                $quotation_id,
                $item['product_id'],
                $item['quantity'],
                $item['unit_price'],
                $item['total_price']
            ]);
        }
        
        // Commit transaction
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Quotation ' . $quotation['quote_number'] . ' updated successfully',
            'quote_number' => $quotation['quote_number'],
            'total_amount' => $total_amount
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?> 